<?php
namespace app\controllers;

use Yii;
use yii\web\Controller;
use yii\filters\AccessControl;
use app\models\Booking;
use app\models\User;
use app\models\RoomType;
use app\models\Review;
use yii\data\ActiveDataProvider;
use yii\web\NotFoundHttpException;

class AdminController extends Controller
{
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    [
                        'allow' => true,
                        'matchCallback' => function ($rule, $action) {
                            return !Yii::$app->user->isGuest && Yii::$app->user->identity->username === 'hotel';
                        },
                        'roles' => ['@'],
                    ],
                ],
                'denyCallback' => function () {
                    throw new \yii\web\ForbiddenHttpException('Доступ запрещен. Только для администратора.');
                }
            ],
        ];
    }

    public function actionIndex()
    {
        $bookingsCount = Booking::find()->count();
        $usersCount = User::find()->count();
        $reviewsCount = Review::find()->count();

        $newBookings = Booking::find()
            ->where(['>=', 'created_at', date('Y-m-d', strtotime('-7 days'))])
            ->count();

        $recentBookings = Booking::find()
            ->with(['user', 'roomType'])
            ->orderBy(['created_at' => SORT_DESC])
            ->limit(10)
            ->all();

        return $this->render('index', [
            'bookingsCount' => $bookingsCount,
            'usersCount' => $usersCount,
            'reviewsCount' => $reviewsCount,
            'newBookings' => $newBookings,
            'recentBookings' => $recentBookings,
        ]);
    }

    public function actionBookings()
    {
        $dataProvider = new ActiveDataProvider([
            'query' => Booking::find()->with(['user', 'roomType']),
            'sort' => [
                'defaultOrder' => ['created_at' => SORT_DESC],
            ],
            'pagination' => [
                'pageSize' => 20,
            ],
        ]);

        return $this->render('bookings', [
            'dataProvider' => $dataProvider,
        ]);
    }

    public function actionUpdateBooking($id)
    {
        $model = $this->findBookingModel($id);

        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            Yii::$app->session->setFlash('success', 'Статус заявки успешно обновлен.');
            return $this->redirect(['bookings']);
        }

        return $this->render('update-booking', [
            'model' => $model,
        ]);
    }

    public function actionViewBooking($id)
    {
        $model = $this->findBookingModel($id);

        return $this->render('view-booking', [
            'model' => $model,
        ]);
    }

    public function actionUsers()
    {
        $dataProvider = new ActiveDataProvider([
            'query' => User::find(),
            'sort' => [
                'defaultOrder' => ['created_at' => SORT_DESC],
            ],
            'pagination' => [
                'pageSize' => 20,
            ],
        ]);

        return $this->render('users', [
            'dataProvider' => $dataProvider,
        ]);
    }

    public function actionReviews()
    {
        $dataProvider = new ActiveDataProvider([
            'query' => Review::find()->with(['user', 'booking']),
            'sort' => [
                'defaultOrder' => ['created_at' => SORT_DESC],
            ],
            'pagination' => [
                'pageSize' => 20,
            ],
        ]);

        return $this->render('reviews', [
            'dataProvider' => $dataProvider,
        ]);
    }

    public function actionRoomTypes()
    {
        $dataProvider = new ActiveDataProvider([
            'query' => RoomType::find(),
            'pagination' => [
                'pageSize' => 10,
            ],
        ]);

        return $this->render('room-types', [
            'dataProvider' => $dataProvider,
        ]);
    }

    public function actionUpdateRoomType($id)
    {
        $model = RoomType::findOne($id);
        if (!$model) {
            throw new NotFoundHttpException('Тип номера не найден.');
        }

        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            Yii::$app->session->setFlash('success', 'Тип номера успешно обновлен.');
            return $this->redirect(['room-types']);
        }

        return $this->render('update-room-type', [
            'model' => $model,
        ]);
    }

    protected function findBookingModel($id)
    {
        if (($model = Booking::findOne($id)) !== null) {
            return $model;
        }

        throw new NotFoundHttpException('Заявка не найдена.');
    }
}