<?php
// models/Booking.php
namespace app\models;

use Yii;
use yii\db\ActiveRecord;

class Booking extends ActiveRecord
{
    const STATUS_NEW = 'new';
    const STATUS_IN_PROCESS = 'in_process';
    const STATUS_COMPLETED = 'completed';
    const STATUS_CANCELLED = 'cancelled';

    const PAYMENT_CASH = 'cash';
    const PAYMENT_CARD = 'card';

    public static function tableName()
    {
        return 'bookings';
    }

    public function rules()
    {
        return [
            [['user_id', 'room_type_id', 'check_in_date', 'check_out_date', 'adults_count', 'payment_method'], 'required'],
            [['user_id', 'room_type_id', 'adults_count', 'children_count'], 'integer'],
            [['check_in_date', 'check_out_date'], 'safe'],
            [['additional_services'], 'string'],
            [['payment_method'], 'in', 'range' => [self::PAYMENT_CASH, self::PAYMENT_CARD]],
            [['status'], 'in', 'range' => [self::STATUS_NEW, self::STATUS_IN_PROCESS, self::STATUS_COMPLETED, self::STATUS_CANCELLED]],
            [['total_amount'], 'number'],
            [['check_out_date'], 'compare', 'compareAttribute' => 'check_in_date', 'operator' => '>', 'message' => 'Дата выезда должна быть позже даты заезда.'],
            [['adults_count', 'children_count'], 'compare', 'compareValue' => 0, 'operator' => '>', 'type' => 'number', 'message' => 'Количество гостей должно быть положительным.'],
        ];
    }

    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'user_id' => 'Пользователь',
            'room_type_id' => 'Тип номера',
            'check_in_date' => 'Дата заезда',
            'check_out_date' => 'Дата выезда',
            'adults_count' => 'Количество взрослых',
            'children_count' => 'Количество детей',
            'additional_services' => 'Дополнительные услуги',
            'payment_method' => 'Способ оплаты',
            'status' => 'Статус',
            'total_amount' => 'Общая сумма',
            'created_at' => 'Создана',
            'updated_at' => 'Обновлена',
        ];
    }

    public function getUser()
    {
        return $this->hasOne(User::class, ['id' => 'user_id']);
    }

    public function getRoomType()
    {
        return $this->hasOne(RoomType::class, ['id' => 'room_type_id']);
    }

    public function getReview()
    {
        return $this->hasOne(Review::class, ['booking_id' => 'id']);
    }

    public function calculateTotal()
    {
        $days = (strtotime($this->check_out_date) - strtotime($this->check_in_date)) / (60 * 60 * 24);
        if ($days <= 0) {
            $days = 1;
        }
        $this->total_amount = $days * $this->roomType->price_per_night;
        return $this->total_amount;
    }

    public function beforeSave($insert)
    {
        if (parent::beforeSave($insert)) {
            if ($insert) {
                $this->calculateTotal();
            }
            return true;
        }
        return false;
    }

    public static function getStatusLabels()
    {
        return [
            self::STATUS_NEW => 'Новая',
            self::STATUS_IN_PROCESS => 'В процессе',
            self::STATUS_COMPLETED => 'Выполнена',
            self::STATUS_CANCELLED => 'Отменена',
        ];
    }

    public function getStatusLabel()
    {
        $statuses = self::getStatusLabels();
        return $statuses[$this->status] ?? $this->status;
    }
}